#!/usr/bin/env python3

# -*- encoding: utf-8; py-indent-offset: 4 -*-

# (c) Michael Honkoop <mhonkoop@comsolve.nl>

# License: GNU General Public License v2

import datetime
import math
from typing import Mapping, Any
from cmk.agent_based.v2 import (
    AgentSection,
    CheckPlugin,
    CheckResult,
    Result,
    State,
    Metric,
    get_value_store,
    check_levels,
)

from cmk_addons.plugins.edirectory_monitor.lib import (
    convert_timestamp,
    discover_edirectory_items,    
    parse_ldap_data,
    ignored_items,
    uptime_attributes,
    time_attributes,
    time_attributes_ignored,

)
'''Definition of keys which hold a non-graphable value'''
non_graphable = [
    "arch",
    "average_load",
    "comitted",
    "configuration",
    "containment",
    "connectedToRemoteLoader",
    "downTime",
    "DriverDN",
    "driverSetDN",
    "driver-state",
    "EngineVersion",
    "id",
    "initial",
    "JobDN",
    "lock",
    "name",
    "nextScheduledRun",
    "numberOfDrivers",
    "processors",
    "running",
    "shutDownPending",
    "starting",
    "startOption",
    "startUpOption_auto-start_count",
    "startUpOption_manual_count",
    "startUpOption_disabled_count",
    "state",
    "status",
    "stopped",
    "total",
    "type",
    "uptime",
    "used",
    "version",
]

'''Definition of keys which hold a total counter'''
total_counters = [
    "nrf_identity",
    "nrf_request",
    "nrf_role",
    "jdbc_statement",
    "publisher_add",
    "publisher_add-association",
    "publisher_check-password",
    "publisher_delete",
    "publisher_generated-password",
    "publisher_get-named-password",
    "publisher_init-params",
    "publisher_instance",
    "publisher_modify",
    "publisher_modify-association",
    "publisher_modify-password",
    "publisher_move",
    "publisher_password",
    "publisher_query",
    "publisher_query-ex",
    "publisher_query-schema",
    "publisher_remove-association",
    "publisher_rename",
    "publisher_schema-def",
    "publisher_service-supported",
    "publisher_status",
    "publisher_sync",
    "publisher_trigger",
    "subscriber_add",
    "subscriber_check-object-password",
    "subscriber_delete",
    "subscriber_modify",
    "subscriber_modify-password",
    "subscriber_move",
    "subscriber_query",
    "subscriber_query-ex",
    "subscriber_rename",
    "subscriber_trigger",   
]

def check_edirectory_items(item: str, params: Mapping[str, Any], section) -> CheckResult:
    value_store = get_value_store()
    '''Split the item key and store the last part for making the returned result and/or metric more unique'''
    item_detail = item.split()[-1]
    data = section.get(item)
    warn, crit = params.get("levels", ("fixed", (None, None)))[1]
    if not data:
        return
    for key, value in data.items():
        if (item_detail + "_" + key) in time_attributes_ignored:
            '''do not iternate keys which only hold a timestamp'''
            continue
        elif key in ignored_items:
            '''do not iternate keys which hold unwanted data'''
            continue
        elif (item_detail + "_" + key) in uptime_attributes:
            '''create readable notation of uptime attributes from seconds'''
            uptime = datetime.timedelta(seconds=int(value))
            yield Result(state=State(0), summary=f"{key}: {uptime}", details=f"{key}: {uptime}")
        elif (item_detail + "_" + key) in time_attributes:
            '''if a timestamp attribute is not ignored, convert it to local time in (human) readable format'''
            datevalue = convert_timestamp(value)
            yield Result(state=State(0), notice=f"{key}: {datevalue}", details=f"{key}: {datevalue}")
        else:
            '''create a default result if above criteria do not apply'''
            partialkey = key.split(".")[0]
            warn, crit = params.get((item.lower()).replace(" ", "_"), {}).get(partialkey, ("fixed", (None, None)))[1]
            if warn is not None or crit is not None:
                if (item_detail + "_" + key) in total_counters:
                    previous_value = value_store.get(key, 0)
                    value_store[key] = value
                    value_difference = int(value) - int(previous_value)
                    yield from check_levels(
                        int(math.ceil(value_difference)),
                        levels_upper=params.get((item.lower()).replace(" ", "_"), {}).get(partialkey, ("fixed", (None, None))),
                        label=key,
                        metric_name=key,
                        notice_only=True,
                        )
                else:
                    yield from check_levels(
                        int(value),
                        levels_upper=params.get((item.lower()).replace(" ", "_"), {}).get(partialkey, ("fixed", (None, None))),
                        label=key,
                        metric_name=key,
                        notice_only=True,
                        )
            else:
                yield Result(state=State(0), notice=f"{key}: {value}", details=f"{key}: {value}")
        if (item_detail + "_" + key) in total_counters:
            '''create a metric which is the difference between previous check value and current check value'''
            previous_value = value_store.get(key, 0)
            value_store[key] = value
            if (previous_value > value):
                previous_value = 0
            value_difference = int(value) - int(previous_value)
            yield Metric((item_detail + "_" + key), abs(value_difference), boundaries=(0, None))
        elif key in non_graphable:
            continue
        else:
            '''Only create a metric for graphable values'''
            yield Metric((item_detail + "_" + key), abs(int(value)))

agent_section_edirectory_monitor_idm = AgentSection(
    name="edirectory_monitor_idm",
    parse_function=parse_ldap_data,
    )

check_plugin_edirectory_monitor_idm = CheckPlugin(
    name="edirectory_monitor_idm",
    service_name="%s",
    discovery_function=discover_edirectory_items,
    check_function=check_edirectory_items,
    check_default_parameters={},
    check_ruleset_name="edirectory_monitor",
)
