#!/usr/bin/env python3
# -*- mode: Python; encoding: utf-8; indent-offset: 4; autowrap: nil -*-

import sys
import ldap
import json
import re

def clean_key(key):
    """Clean and format the key by removing unwanted parts."""
    cleaned_key = ' '.join(key.split(',')[::-1]).replace('cn=Monitor', '').replace('cn=', '').strip()
    return cleaned_key

def clean_value(value):
    cleaned_value = value.replace(" Bytes", "").replace(" KB", "")
    return cleaned_value

def print_sections(raw_result):
    separator = 124 # '|'; for more fun use separator=7 (which is a bell)
    lines = []
    for each_item in raw_result:
        if isinstance(each_item, list):
            print_sections(each_item)
        else:
            if "BackGroundProcInterval" in each_item[0]:
                continue
            lines.append(clean_key(each_item[0]))
            for k,v in each_item[1].items():
                if k  == "objectclass":
                    continue
                decoded_value = v[0].decode("utf-8")
                if "CheckPointThreadForceStartTime" in k or "CheckPointThreadStartTime" in k:
                    continue
                lines.append(chr(separator).join(['{}={}'.format(k,clean_value(decoded_value))]))

    if lines:
        print('<<<edirectory_monitor:sep({})>>>')
        print(chr(separator).join(lines))


def print_list(the_list):
    for each_item in the_list:
        if isinstance(each_item, list):
            print_list(each_item)
        else:
            print(clean_key(each_item[0]))

def process_entry(entry):
    resultArray = ['0 "eDirectory ']
    detailArray = []


def main():
    args = sys.argv[1:]
    if len(args) < 3:
        print("Usage: script.py <LDAP_URI> <BIND_DN> <PASSWORD>")
        sys.exit(1)

    ldap_uri = args[0]
    binddn = args[1] 
    pw = args[2]
    basedn = "cn=Monitor"
    searchFilter = "(objectClass=*)"
    searchAttribute = ["*"]
    searchScope = ldap.SCOPE_SUBTREE

    try:
        # ignore TLS certificate checking
        ldap.set_option(ldap.OPT_X_TLS_REQUIRE_CERT, ldap.OPT_X_TLS_NEVER)
        # Create LDAPObject instance with given uri
        l = ldap.initialize(ldap_uri)
        # Set LDAP protocol version used
        l.protocol_version = ldap.VERSION3
    except ldap.LDAPError as e:
        print(f"Failed to initialize LDAP connection: {e}")
        sys.exit(1)

    try:
        # Attempt to bind with given credentials
        l.simple_bind_s(binddn, pw)
    except ldap.INVALID_CREDENTIALS:
          print("Authentication to the LDAP host has failed.")
          sys.exit(1)
    except ldap.LDAPError as e:
        print(f"LDAP error during bind: {e}")
        sys.exit(1)

    try:
        ldap_result_id = l.search(basedn, searchScope, searchFilter, searchAttribute)
        result_set = []

        while True:
            result_type, result_data = l.result(ldap_result_id, 0)
            if (result_data == []):
                break
            elif result_type == ldap.RES_SEARCH_ENTRY:
                result_set.append(result_data)

        for i in range(len(result_set)):
            print_sections(result_set[i])

    except ldap.LDAPError as e:
        print(f"LDAP search failed: {e}")

    finally:
        l.unbind_s()

if __name__ == "__main__":
    main()

