#!/usr/bin/env python3

# -*- encoding: utf-8; py-indent-offset: 4 -*-

# (c) Michael Honkoop <mhonkoop@comsolve.nl>

# License: GNU General Public License v2

import datetime
import math
from typing import Mapping, Any
from cmk.agent_based.v2 import (
    AgentSection,
    CheckPlugin,
    CheckResult,
    Result,
    State,
    Metric,
    get_value_store,
    check_levels,
)

from cmk_addons.plugins.edirectory_monitor.lib import (
    convert_timestamp,
    discover_edirectory_items,    
    parse_ldap_data,
    ignored_items,
    uptime_attributes,
    time_attributes,
    time_attributes_ignored,
    non_graphable_attributes,
    total_counter_attributes,
    idm_nongraphable_attributes,
)

def check_edirectory_items(item: str, params: Mapping[str, Any], section) -> CheckResult:
    value_store = get_value_store()
    '''Split the item key and store the last part for making the returned result and/or metric more unique'''
    item_detail = item.split()[-1]
    data = section.get(item)
    warn, crit = params.get("levels", ("fixed", (None, None)))[1]
    # The warn and crit you can use than later inside a check_levels function
     # print(f"Levels - {warn} - {crit}")
    if not data:
        return
    for key, value in data.items():
        if (item_detail + "_" + key) in time_attributes_ignored:
            '''do not iternate keys which only hold a timestamp'''
            continue
        elif key in ignored_items:
            '''do not iternate keys which hold unwanted data'''
            continue
        elif (item_detail + "_" + key) in uptime_attributes:
            '''create readable notation of uptime attributes from seconds'''
            uptime = datetime.timedelta(seconds=int(value))
            yield Result(state=State(0), summary=f"{key}: {uptime}", details=f"{key}: {uptime}")
        elif (item_detail + "_" + key) in time_attributes:
            '''if a timestamp attribute is not ignored, convert it to local time in (human) readable format'''
            datevalue = convert_timestamp(value)
            yield Result(state=State(0), summary=f"{key}: {datevalue}", details=f"{key}: {datevalue}")
        else:
            '''create a default result if above criteria do not apply'''
            warn, crit = params.get((item.lower()).replace(" ", "_"), {}).get(key, ("fixed", (None, None)))[1]
            if warn is not None or crit is not None:
                #print(f"Levels {key} - {warn} - {crit}")
                if (item_detail + "_" + key) in total_counter_attributes:
                    previous_value = value_store.get(key, 0)
                    value_store[key] = value
                    value_difference = int(value) - int(previous_value)
                    yield from check_levels(
                        int(math.ceil(value_difference)),
                        levels_upper=params.get((item.lower()).replace(" ", "_"), {}).get(key, ("fixed", (None, None))),
                        label=key,
                        metric_name=key,
                        notice_only=True,
                        )
                else:
                    yield from check_levels(
                        int(value),
                        levels_upper=params.get((item.lower()).replace(" ", "_"), {}).get(key, ("fixed", (None, None))),
                        label=key,
                        metric_name=key,
                        notice_only=True,
                        )
            else:
                yield Result(state=State(0), notice=f"{key}: {value}", details=f"{key}: {value}")
        if (item_detail + "_" + key) in total_counter_attributes:
            '''create a metric which is the difference between previous check value and current check value'''
            previous_value = value_store.get(key, 0)
            value_store[key] = value
            value_difference = int(value) - int(previous_value)
            yield Metric((item_detail + "_" + key), abs(value_difference), boundaries=(0, None))
        elif item.startswith("IDM") and key in idm_nongraphable_attributes:
            continue
        elif (item_detail + "_" + key) not in non_graphable_attributes:
            '''Only create a metric for graphable values'''
            yield Metric((item_detail + "_" + key), abs(int(value)))

agent_section_edirectory_monitor_dhost = AgentSection(
    name="edirectory_monitor_dhost",
    parse_function=parse_ldap_data,
    )

check_plugin_edirectory_monitor_dhost = CheckPlugin(
    name="edirectory_monitor_dhost",
    service_name="%s",
    discovery_function=discover_edirectory_items,
    check_function=check_edirectory_items,
    check_default_parameters={},
    check_ruleset_name="edirectory_monitor_dhost",
)
